'use strict';

import MetaApi, {MetatraderAccount, RpcMetaApiConnectionInstance, StreamingMetaApiConnectionInstance} from '../index';
import * as helpers from '../helpers/helpers';
import log4js from 'log4js';

require('dotenv').config();
const token = process.env.TOKEN;
const login = process.env.LOGIN;
const password = process.env.PASSWORD;
const serverName = process.env.SERVER;

describe('Refreshing terminal state integration', function() {
  this.timeout(1000 * 60 * 10);

  if (!token) {
    return;
  }

  MetaApi.enableLog4jsLogging();
  const api = new MetaApi(token, {domain: 'agiliumtrade.gfyt.agiliumlabs.cloud'});
  const logger = log4js.getLogger('test');
  
  /** @type {MetatraderAccount} */
  let account;

  before(async () => {
    log4js.configure(helpers.assembleLog4jsConfig({
      levels: {
        'TerminalState': 'TRACE'
      }
    }));
    account = await prepareAccount();
  });

  after(async () => {
    api.close();
  });

  describe('streaming connection', () => {

    /** @type {StreamingMetaApiConnectionInstance} */
    let connection;

    before(async () => {
      connection = account.getStreamingConnection();
      await connection.connect();
      await connection.waitSynchronized();
    });

    after(async () => {
      await connection.close();
    });

    it('should refresh terminal state with streaming api', async () => {
      await connection.subscribeToMarketData('BTCUSD', [{type: 'quotes'}]);
      await connection.terminalState.refreshTerminalState();
    });

  });

  describe('rpc connection', () => {

    /** @type {RpcMetaApiConnectionInstance} */
    let connection;

    before(async () => {
      connection = account.getRPCConnection();
      await connection.connect();
      await connection.waitSynchronized();
    });

    after(async () => {
      await connection.close();
    });

    it('should retrieve refreshed symbol quotes', async () => {
      let refreshedQuotes = await connection.refreshSymbolQuotes(['BTCUSD']);
      logger.info('Received refreshed quotes', JSON.stringify(refreshedQuotes));
    });

    it('should retrieve account information error with refreshing terminal state', async () => {
      let accountInformation = await connection.getAccountInformation({refreshTerminalState: true});
      logger.info('Received account information', JSON.stringify(accountInformation));
    });

  });

  /**
   * Prepares metatrader account for test
   * @returns {Promise<MetatraderAccount>} promise resolving with test account 
   */
  async function prepareAccount() {
    let accounts = await api.metatraderAccountApi.getAccountsWithInfiniteScrollPagination();
    let result = accounts.find(a => a.login === login && a.type.startsWith('cloud-g2'));
    if (!result) {
      result = await api.metatraderAccountApi.createAccount({
        name: 'Test account',
        type: 'cloud-g2',
        login: login,
        password: password,
        server: serverName,
        application: 'MetaApi',
        magic: 1000
      });
    }
    await result.deploy();
    await result.waitConnected();
    return result;
  }

});
