"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return TerminalState;
    }
});
const _randomstring = /*#__PURE__*/ _interop_require_default(require("randomstring"));
const _synchronizationListener = /*#__PURE__*/ _interop_require_default(require("../clients/metaApi/synchronizationListener"));
const _metaApiWebsocketclient = /*#__PURE__*/ _interop_require_default(require("../clients/metaApi/metaApiWebsocket.client"));
const _logger = /*#__PURE__*/ _interop_require_default(require("../logger"));
const _terminalHashManager = /*#__PURE__*/ _interop_require_default(require("./terminalHashManager"));
const _metatraderAccount = /*#__PURE__*/ _interop_require_default(require("./metatraderAccount"));
const _promises = require("../helpers/promises");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let TerminalState = class TerminalState extends _synchronizationListener.default {
    get id() {
        return this._id;
    }
    /**
   * Returns true if MetaApi have connected to MetaTrader terminal
   * @return {Boolean} true if MetaApi have connected to MetaTrader terminal
   */ get connected() {
        return Object.values(this._stateByInstanceIndex).reduce((acc, s)=>acc || s.connected, false);
    }
    /**
   * Returns true if MetaApi have connected to MetaTrader terminal and MetaTrader terminal is connected to broker
   * @return {Boolean} true if MetaApi have connected to MetaTrader terminal and MetaTrader terminal is connected to
   * broker
   */ get connectedToBroker() {
        return Object.values(this._stateByInstanceIndex).reduce((acc, s)=>acc || s.connectedToBroker, false);
    }
    /**
   * Returns a local copy of account information
   * @returns {MetatraderAccountInformation} local copy of account information
   */ get accountInformation() {
        return this._combinedState.accountInformation;
    }
    /**
   * Returns a local copy of MetaTrader positions opened
   * @returns {Array<MetatraderPosition>} a local copy of MetaTrader positions opened
   */ get positions() {
        const hash = this._combinedState.positionsHash;
        return hash ? Object.values(this._terminalHashManager.getPositionsByHash(hash) || {}) : [];
    }
    /**
   * Returns a local copy of MetaTrader orders opened
   * @returns {Array<MetatraderOrder>} a local copy of MetaTrader orders opened
   */ get orders() {
        const hash = this._combinedState.ordersHash;
        return hash ? Object.values(this._terminalHashManager.getOrdersByHash(hash) || {}) : [];
    }
    /**
   * Returns a local copy of symbol specifications available in MetaTrader trading terminal
   * @returns {Array<MetatraderSymbolSpecification>} a local copy of symbol specifications available in MetaTrader
   * trading terminal
   */ get specifications() {
        const hash = this._combinedState.specificationsHash;
        return hash ? Object.values(this._terminalHashManager.getSpecificationsByHash(this._combinedState.specificationsHash) || {}) : [];
    }
    /**
   * Returns hashes of terminal state data for incremental synchronization
   * @returns {Promise<Object>} promise resolving with hashes of terminal state data
   */ // eslint-disable-next-line complexity
    getHashes() {
        const specificationsHashes = this._terminalHashManager.getLastUsedSpecificationHashes(this._account.server);
        const positionsHashes = this._terminalHashManager.getLastUsedPositionHashes(this._account.id);
        const ordersHashes = this._terminalHashManager.getLastUsedOrderHashes(this._account.id);
        return {
            specificationsHashes: specificationsHashes,
            positionsHashes: positionsHashes,
            ordersHashes: ordersHashes
        };
    }
    /**
   * Returns MetaTrader symbol specification by symbol
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @return {MetatraderSymbolSpecification} MetatraderSymbolSpecification found or undefined if specification for a
   * symbol is not found
   */ specification(symbol) {
        if (this._combinedState.specificationsHash) {
            const state = this._terminalHashManager.getSpecificationsByHash(this._combinedState.specificationsHash);
            return state[symbol];
        } else {
            return null;
        }
    }
    /**
   * Returns MetaTrader symbol price by symbol
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @return {MetatraderSymbolPrice} MetatraderSymbolPrice found or undefined if price for a symbol is not found
   */ price(symbol) {
        return this._combinedState.pricesBySymbol[symbol];
    }
    /**
   * Quote time
   * @typdef {Object} QuoteTime
   * @property {Date} time quote time
   * @property {String} brokerTime quote time in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   */ /**
   * Returns time of the last received quote
   * @return {QuoteTime} time of the last received quote
   */ get lastQuoteTime() {
        if (this._combinedState.lastQuoteTime) {
            return {
                time: this._combinedState.lastQuoteTime,
                brokerTime: this._combinedState.lastQuoteBrokerTime
            };
        } else {
            return undefined;
        }
    }
    /**
   * Waits for price to be received
   * @param {string} symbol symbol (e.g. currency pair or an index)
   * @param {number} [timeoutInSeconds] timeout in seconds, default is 30
   * @return {Promise<MetatraderSymbolPrice>} promise resolving with price or undefined if price has not been received
   */ async waitForPrice(symbol, timeoutInSeconds = 30) {
        this._waitForPriceResolves[symbol] = this._waitForPriceResolves[symbol] || [];
        if (!this.price(symbol)) {
            await Promise.race([
                new Promise((res)=>this._waitForPriceResolves[symbol].push(res)),
                new Promise((res)=>setTimeout(res, timeoutInSeconds * 1000))
            ]);
        }
        return this.price(symbol);
    }
    /**
   * Invoked when connection to MetaTrader terminal established
   * @param {String} instanceIndex index of an account instance connected
   */ onConnected(instanceIndex) {
        this._getState(instanceIndex).connected = true;
    }
    /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {String} instanceIndex index of an account instance connected
   */ onDisconnected(instanceIndex) {
        let state = this._getState(instanceIndex);
        state.connected = false;
        state.connectedToBroker = false;
    }
    /**
   * Invoked when broker connection status have changed
   * @param {String} instanceIndex index of an account instance connected
   * @param {Boolean} connected is MetaTrader terminal is connected to broker
   */ onBrokerConnectionStatusChanged(instanceIndex, connected) {
        this._combinedState.lastStatusTime = Date.now();
        this._getState(instanceIndex).connectedToBroker = connected;
    }
    /**
   * Invoked when MetaTrader terminal state synchronization is started
   * @param {string} instanceIndex index of an account instance connected
   * @param {string} specificationsHash specifications hash
   * @param {string} positionsHash positions hash
   * @param {string} ordersHash orders hash
   * @param {string} synchronizationId synchronization id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onSynchronizationStarted(instanceIndex, specificationsHash, positionsHash, ordersHash, synchronizationId) {
        const unsynchronizedStates = this._getStateIndicesOfSameInstanceNumber(instanceIndex).filter((stateIndex)=>!this._stateByInstanceIndex[stateIndex].ordersInitialized);
        unsynchronizedStates.sort((a, b)=>b.lastSyncUpdateTime - a.lastSyncUpdateTime);
        unsynchronizedStates.slice(1).forEach((stateIndex)=>this._removeState(stateIndex));
        let state = this._getState(instanceIndex);
        state.isSpecificationsExpected = !specificationsHash;
        state.isPositionsExpected = !positionsHash;
        state.isOrdersExpected = !ordersHash;
        state.lastSyncUpdateTime = Date.now();
        state.accountInformation = undefined;
        state.pricesBySymbol = {};
        state.positions = [];
        if (!positionsHash) {
            state.positionsInitialized = false;
            state.positionsHash = null;
        } else {
            state.positionsHash = positionsHash;
        }
        state.orders = [];
        if (!ordersHash) {
            state.ordersInitialized = false;
            state.ordersHash = null;
        } else {
            state.ordersHash = ordersHash;
        }
        state.specificationsBySymbol = {};
        if (!specificationsHash) {
            this._logger.trace(()=>`${this._account.id}:${instanceIndex}:${synchronizationId}: cleared specifications ` + "on synchronization start");
            state.specificationsHash = null;
        } else {
            this._logger.trace(()=>`${this._account.id}:${instanceIndex}:${synchronizationId}: no need to clear ` + `specifications on synchronization start, ${Object.keys(state.specificationsBySymbol || {}).length} ` + "specifications reused");
            state.specificationsHash = specificationsHash;
        }
    }
    /**
   * Invoked when MetaTrader account information is updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderAccountInformation} accountInformation updated MetaTrader account information
   */ onAccountInformationUpdated(instanceIndex, accountInformation) {
        let state = this._getState(instanceIndex);
        this._refreshStateUpdateTime(instanceIndex);
        state.accountInformation = accountInformation;
        if (accountInformation) {
            this._combinedState.accountInformation = Object.assign({}, accountInformation);
        }
    }
    /**
   * Invoked when the positions are replaced as a result of initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderPosition>} positions updated array of positions
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onPositionsReplaced(instanceIndex, positions) {
        let state = this._getState(instanceIndex);
        this._refreshStateUpdateTime(instanceIndex);
        if (state.isPositionsExpected) {
            state.positions = positions;
        }
    }
    /**
   * Invoked when position synchronization fnished to indicate progress of an initial terminal state synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onPositionsSynchronized(instanceIndex, synchronizationId) {
        let state = this._getState(instanceIndex);
        state.positionsInitialized = true;
    }
    /**
   * Invoked when MetaTrader positions are updated
   * @param {string} instanceIndex index of an account instance connected
   * @param {MetatraderPosition[]} positions updated MetaTrader positions
   * @param {string[]} removedPositionIds removed position ids
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ async onPositionsUpdated(instanceIndex, positions, removedPositionIds) {
        let instanceState = this._getState(instanceIndex);
        this._refreshStateUpdateTime(instanceIndex);
        const date = Date.now();
        removedPositionIds.forEach((id)=>this._combinedState.removedPositions[id] = date);
        positions = this._filterRemovedPositions(positions);
        Object.keys(this._combinedState.removedPositions).forEach((id)=>{
            if (this._combinedState.removedPositions[id] < date - 24 * 60 * 60 * 1000) {
                delete this._combinedState.removedPositions[id];
            }
        });
        if (instanceState.ordersInitialized) {
            const updatePositions = async (state, instance)=>{
                const hash = await this._terminalHashManager.updatePositions(this._account.id, this._account.type, this._id, instance, positions, removedPositionIds, state.positionsHash);
                state.positionsHash = hash;
            };
            await updatePositions(instanceState, instanceIndex);
            await updatePositions(this._combinedState, this._combinedInstanceIndex);
        } else {
            instanceState.positions = instanceState.positions.filter((position)=>!removedPositionIds.includes(position.id));
            positions.forEach((position)=>{
                let index = instanceState.positions.findIndex((p)=>p.id === position.id);
                if (index !== -1) {
                    instanceState.positions[index] = position;
                } else {
                    instanceState.positions.push(position);
                }
            });
        }
    }
    /**
   * Invoked when the orders are replaced as a result of initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderOrder>} orders updated array of pending orders
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onPendingOrdersReplaced(instanceIndex, orders) {
        let state = this._getState(instanceIndex);
        this._refreshStateUpdateTime(instanceIndex);
        if (state.isOrdersExpected) {
            state.orders = orders;
        }
    }
    /**
   * Invoked when pending order synchronization fnished to indicate progress of an initial terminal state
   * synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ // eslint-disable-next-line complexity, max-statements
    async onPendingOrdersSynchronized(instanceIndex, synchronizationId) {
        let state = this._getState(instanceIndex);
        state.positionsInitialized = true;
        state.ordersInitialized = true;
        this._combinedState.accountInformation = state.accountInformation ? Object.assign({}, state.accountInformation) : undefined;
        state.positions = this._filterRemovedPositions(state.positions);
        if (state.positions.length) {
            const hash = this._terminalHashManager.recordPositions(this._account.id, this._account.type, this._id, instanceIndex, state.positions);
            state.positionsHash = hash;
            this._combinedState.positions = (state.positions || []).map((p)=>Object.assign({}, p));
            this._combinedState.positionsHash = hash;
        } else if (state.positionsHash) {
            this._terminalHashManager.removePositionReference(this.id, instanceIndex);
            this._terminalHashManager.addPositionReference(state.positionsHash, this.id, instanceIndex);
            this._combinedState.positionsHash = state.positionsHash;
            this._terminalHashManager.removePositionReference(this.id, this._combinedInstanceIndex);
            this._terminalHashManager.addPositionReference(state.positionsHash, this.id, this._combinedInstanceIndex);
        }
        state.orders = this._filterRemovedOrders(state.orders);
        if (state.orders.length) {
            const hash = this._terminalHashManager.recordOrders(this._account.id, this._account.type, this._id, instanceIndex, state.orders);
            state.ordersHash = hash;
            this._combinedState.orders = (state.orders || []).map((o)=>Object.assign({}, o));
            this._combinedState.ordersHash = hash;
        } else if (state.ordersHash) {
            this._terminalHashManager.removeOrderReference(this.id, instanceIndex);
            this._terminalHashManager.addOrderReference(state.ordersHash, this.id, instanceIndex);
            this._combinedState.ordersHash = state.ordersHash;
            this._terminalHashManager.removeOrderReference(this.id, this._combinedInstanceIndex);
            this._terminalHashManager.addOrderReference(state.ordersHash, this.id, this._combinedInstanceIndex);
        }
        this._logger.trace(()=>`${this._account.id}:${instanceIndex}:${synchronizationId}: assigned specifications to ` + "combined state from " + `${instanceIndex}, ${Object.keys(state.specificationsBySymbol || {}).length} specifications assigned`);
        this._combinedState.positionsInitialized = true;
        this._combinedState.ordersInitialized = true;
        if (Object.keys(state.specificationsBySymbol || {}).length) {
            if (state.isSpecificationsExpected) {
                const hash = await this._terminalHashManager.recordSpecifications(this._account.server, this._account.type, this._id, instanceIndex, Object.values(state.specificationsBySymbol));
                this._combinedState.specificationsHash = hash;
                state.specificationsHash = hash;
                state.specificationsBySymbol = null;
            } else if (state.specificationsHash) {
                const hash = await this._terminalHashManager.updateSpecifications(this._account.server, this._account.type, this._id, instanceIndex, Object.values(state.specificationsBySymbol), [], state.specificationsHash);
                state.specificationsHash = hash;
            }
        } else if (state.specificationsHash) {
            this._terminalHashManager.removeSpecificationReference(this.id, instanceIndex);
            this._terminalHashManager.addSpecificationReference(state.specificationsHash, this.id, instanceIndex);
            this._combinedState.specificationsHash = state.specificationsHash;
            this._terminalHashManager.removeSpecificationReference(this.id, this._combinedInstanceIndex);
            this._terminalHashManager.addSpecificationReference(state.specificationsHash, this.id, this._combinedInstanceIndex);
        }
        for (let stateIndex of this._getStateIndicesOfSameInstanceNumber(instanceIndex)){
            if (!this._stateByInstanceIndex[stateIndex].connected) {
                this._removeState(stateIndex);
            }
        }
    }
    /**
   * Invoked when MetaTrader pending orders are updated or completed
   * @param {string} instanceIndex index of an account instance connected
   * @param {MetatraderOrder[]} orders updated MetaTrader pending orders
   * @param {string[]} completedOrderIds completed MetaTrader pending order ids
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ async onPendingOrdersUpdated(instanceIndex, orders, completedOrderIds) {
        let instanceState = this._getState(instanceIndex);
        this._refreshStateUpdateTime(instanceIndex);
        const date = Date.now();
        completedOrderIds.forEach((id)=>this._combinedState.completedOrders[id] = date);
        orders = this._filterRemovedOrders(orders);
        Object.keys(this._combinedState.completedOrders).forEach((id)=>{
            if (this._combinedState.completedOrders[id] < date - 24 * 60 * 60 * 1000) {
                delete this._combinedState.completedOrders[id];
            }
        });
        if (instanceState.ordersInitialized) {
            const updatePendingOrders = async (state, instance)=>{
                const hash = await this._terminalHashManager.updateOrders(this._account.id, this._account.type, this._id, instance, orders, completedOrderIds, state.ordersHash);
                state.ordersHash = hash;
            };
            await updatePendingOrders(instanceState, instanceIndex);
            await updatePendingOrders(this._combinedState, this._combinedInstanceIndex);
        } else {
            instanceState.orders = instanceState.orders.filter((order)=>!completedOrderIds.includes(order.id));
            orders.forEach((order)=>{
                let index = instanceState.orders.findIndex((o)=>o.id === order.id);
                if (index !== -1) {
                    instanceState.orders[index] = order;
                } else {
                    instanceState.orders.push(order);
                }
            });
        }
    }
    /**
   * Invoked when a symbol specification was updated
   * @param {String} instanceIndex index of account instance connected
   * @param {Array<MetatraderSymbolSpecification>} specifications updated specifications
   * @param {Array<String>} removedSymbols removed symbols
   */ async onSymbolSpecificationsUpdated(instanceIndex, specifications, removedSymbols) {
        let instanceState = this._getState(instanceIndex);
        this._refreshStateUpdateTime(instanceIndex);
        if (!instanceState.ordersInitialized) {
            for (let specification of specifications){
                instanceState.specificationsBySymbol[specification.symbol] = specification;
            }
        } else {
            const hash = this._terminalHashManager.updateSpecifications(this._account.server, this._account.type, this._id, instanceIndex, specifications, removedSymbols, instanceState.specificationsHash);
            instanceState.specificationsHash = hash;
            const combinedHash = this._terminalHashManager.updateSpecifications(this._account.server, this._account.type, this._id, this._combinedInstanceIndex, specifications, removedSymbols, this._combinedState.specificationsHash);
            this._combinedState.specificationsHash = combinedHash;
        }
        this._logger.trace(()=>`${this._account.id}:${instanceIndex}: updated ${specifications.length} specifications, ` + `removed ${removedSymbols.length} specifications. There are ` + `${Object.keys(instanceState.specificationsBySymbol || {}).length} specifications after update`);
    }
    /**
   * Invoked when prices for several symbols were updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderSymbolPrice>} prices updated MetaTrader symbol prices
   * @param {Number} equity account liquidation value
   * @param {Number} margin margin used
   * @param {Number} freeMargin free margin
   * @param {Number} marginLevel margin level calculated as % of equity/margin
   */ // eslint-disable-next-line complexity
    onSymbolPricesUpdated(instanceIndex, prices, equity, margin, freeMargin, marginLevel) {
        let instanceState = this._getState(instanceIndex);
        this._refreshStateUpdateTime(instanceIndex);
        // eslint-disable-next-line complexity,max-statements
        const updateSymbolPrices = (state)=>{
            let pricesInitialized = false;
            let priceUpdated = false;
            for (let price of prices || []){
                let currentPrice = state.pricesBySymbol[price.symbol];
                if (currentPrice && currentPrice.time.getTime() > price.time.getTime()) {
                    continue;
                } else {
                    priceUpdated = true;
                }
                if (!state.lastQuoteTime || state.lastQuoteTime.getTime() < price.time.getTime()) {
                    state.lastQuoteTime = price.time;
                    state.lastQuoteBrokerTime = price.brokerTime;
                }
                state.pricesBySymbol[price.symbol] = price;
                const allPositions = Object.values(this._terminalHashManager.getPositionsByHash(state.positionsHash) || {});
                const allOrders = Object.values(this._terminalHashManager.getOrdersByHash(state.ordersHash) || {});
                let positions = allPositions.filter((p)=>p.symbol === price.symbol);
                let otherPositions = allPositions.filter((p)=>p.symbol !== price.symbol);
                let orders = allOrders.filter((o)=>o.symbol === price.symbol);
                pricesInitialized = true;
                for (let position of otherPositions){
                    let p = state.pricesBySymbol[position.symbol];
                    if (p) {
                        if (position.unrealizedProfit === undefined) {
                            this._updatePositionProfits(position, p);
                        }
                    } else {
                        pricesInitialized = false;
                    }
                }
                for (let position of positions){
                    this._updatePositionProfits(position, price);
                }
                for (let order of orders){
                    order.currentPrice = order.type === "ORDER_TYPE_BUY" || order.type === "ORDER_TYPE_BUY_LIMIT" || order.type === "ORDER_TYPE_BUY_STOP" || order.type === "ORDER_TYPE_BUY_STOP_LIMIT" ? price.ask : price.bid;
                }
                let priceResolves = this._waitForPriceResolves[price.symbol] || [];
                if (priceResolves.length) {
                    for (let resolve of priceResolves){
                        resolve();
                    }
                    delete this._waitForPriceResolves[price.symbol];
                }
            }
            if (priceUpdated && state.accountInformation) {
                const positions = Object.values(this._terminalHashManager.getPositionsByHash(state.positionsHash) || {});
                if (state.positionsInitialized && pricesInitialized) {
                    if (state.accountInformation.platform === "mt5") {
                        state.accountInformation.equity = equity !== undefined ? equity : state.accountInformation.balance + positions.reduce((acc, p)=>acc + Math.round((p.unrealizedProfit || 0) * 100) / 100 + Math.round((p.swap || 0) * 100) / 100, 0);
                    } else {
                        state.accountInformation.equity = equity !== undefined ? equity : state.accountInformation.balance + positions.reduce((acc, p)=>acc + Math.round((p.swap || 0) * 100) / 100 + Math.round((p.commission || 0) * 100) / 100 + Math.round((p.unrealizedProfit || 0) * 100) / 100, 0);
                    }
                    state.accountInformation.equity = Math.round(state.accountInformation.equity * 100) / 100;
                } else {
                    state.accountInformation.equity = equity !== undefined ? equity : state.accountInformation.equity;
                }
                var _prices__accountCurrencyExchangeRate;
                state.accountInformation.accountCurrencyExchangeRate = (_prices__accountCurrencyExchangeRate = prices[0].accountCurrencyExchangeRate) !== null && _prices__accountCurrencyExchangeRate !== void 0 ? _prices__accountCurrencyExchangeRate : state.accountInformation.accountCurrencyExchangeRate;
                state.accountInformation.margin = margin !== undefined ? margin : state.accountInformation.margin;
                state.accountInformation.freeMargin = freeMargin !== undefined ? freeMargin : state.accountInformation.freeMargin;
                state.accountInformation.marginLevel = freeMargin !== undefined ? marginLevel : state.accountInformation.marginLevel;
            }
        };
        updateSymbolPrices(instanceState);
        updateSymbolPrices(this._combinedState);
        for (let price of prices){
            for (let call of Object.values(this._processThrottledQuotesCalls)){
                var _call_expectedSymbols;
                this._logger.trace(`${this._account.id}:${instanceIndex}: refreshed ${price.symbol} price`);
                (_call_expectedSymbols = call.expectedSymbols) === null || _call_expectedSymbols === void 0 ? void 0 : _call_expectedSymbols.delete(price.symbol);
                call.receivedSymbols.add(price.symbol);
                call.promise.check();
            }
        }
    }
    /**
   * Invoked when a stream for an instance index is closed
   * @param {String} instanceIndex index of an account instance connected
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ async onStreamClosed(instanceIndex) {
        if (this._stateByInstanceIndex[instanceIndex]) {
            for (let stateIndex of this._getStateIndicesOfSameInstanceNumber(instanceIndex)){
                const instanceState = this._stateByInstanceIndex[stateIndex];
                if (!this._stateByInstanceIndex[instanceIndex].ordersInitialized && this._stateByInstanceIndex[instanceIndex].lastSyncUpdateTime <= instanceState.lastSyncUpdateTime) {
                    this._removeState(instanceIndex);
                    break;
                }
                if (instanceState.connected && instanceState.ordersInitialized) {
                    this._removeState(instanceIndex);
                    break;
                }
            }
        }
    }
    /**
   * Forces refresh of most recent quote updates for symbols subscribed to by the terminal, and waits for them all to
   * be processed by this terminal state. This method does not waits for all other listeners to receive and process the
   * quote updates
   * @param {RefreshTerminalStateOptions} [options] additional options
   * @returns {Promise} promise resolving when the terminal state received and processed the latest quotes
   */ async refreshTerminalState(options) {
        let callData = {
            receivedSymbols: new Set()
        };
        let callId = _randomstring.default.generate(8);
        this._processThrottledQuotesCalls[callId] = callData;
        callData.promise = new _promises.ConditionPromise(()=>callData.expectedSymbols && !callData.expectedSymbols.size);
        var _options_timeoutInSeconds;
        callData.promise.timeout(1000 * ((_options_timeoutInSeconds = options === null || options === void 0 ? void 0 : options.timeoutInSeconds) !== null && _options_timeoutInSeconds !== void 0 ? _options_timeoutInSeconds : 10), "refreshing terminal state timed out");
        try {
            let symbols = await Promise.race([
                this._websocketClient.refreshTerminalState(this._account.id),
                callData.promise // will only throw timeout error at this point
            ]);
            this._logger.debug(`${this._account.id}: expecting for ${symbols.length ? symbols : 0} symbols to refresh`);
            let expectedSymbols = new Set();
            for (let symbol of symbols){
                if (!callData.receivedSymbols.has(symbol)) {
                    expectedSymbols.add(symbol);
                }
            }
            callData.expectedSymbols = expectedSymbols;
            callData.promise.check();
            await callData.promise;
        } finally{
            delete this._processThrottledQuotesCalls[callId];
        }
    }
    /**
   * Removes connection related data from terminal hash manager
   */ close() {
        clearInterval(this._checkCombinedStateActivityJobInterval);
        Object.keys(this._stateByInstanceIndex).forEach((instanceIndex)=>{
            this._removeFromHashManager(instanceIndex);
        });
        this._removeFromHashManager(this._combinedInstanceIndex);
    }
    // resets combined state and removes from hash manager if has been disconnected for a long time
    _checkCombinedStateActivityJob() {
        if (!this.connectedToBroker && this._combinedState.lastStatusTime < Date.now() - 30 * 60 * 1000) {
            this._removeFromHashManager(this._combinedInstanceIndex);
            this._combinedState.accountInformation = undefined;
            this._combinedState.specificationsBySymbol = null;
            this._combinedState.pricesBySymbol = {};
            this._combinedState.specificationsHash = null;
            this._combinedState.orders = [];
            this._combinedState.ordersHash = null;
            this._combinedState.positions = [];
            this._combinedState.positionsHash = null;
            this._combinedState.ordersInitialized = false;
            this._combinedState.positionsInitialized = false;
            this._combinedState.lastStatusTime = 0;
            this._combinedState.lastQuoteTime = undefined;
            this._combinedState.lastQuoteBrokerTime = undefined;
        }
    }
    _removeState(instanceIndex) {
        delete this._stateByInstanceIndex[instanceIndex];
        this._removeFromHashManager(instanceIndex);
    }
    _removeFromHashManager(instanceIndex) {
        this._terminalHashManager.removeConnectionReferences(this._account.server, this._account.id, this._id, instanceIndex);
    }
    _refreshStateUpdateTime(instanceIndex) {
        const state = this._stateByInstanceIndex[instanceIndex];
        if (state && state.ordersInitialized) {
            state.lastSyncUpdateTime = Date.now();
        }
    }
    _getStateIndicesOfSameInstanceNumber(instanceIndex) {
        const region = instanceIndex.split(":")[0];
        const instanceNumber = instanceIndex.split(":")[1];
        return Object.keys(this._stateByInstanceIndex).filter((stateInstanceIndex)=>stateInstanceIndex.startsWith(`${region}:${instanceNumber}:`) && instanceIndex !== stateInstanceIndex);
    }
    // eslint-disable-next-line complexity
    _updatePositionProfits(position, price) {
        let specification = this.specification(position.symbol);
        if (specification) {
            let multiplier = Math.pow(10, specification.digits);
            if (position.profit !== undefined) {
                position.profit = Math.round(position.profit * multiplier) / multiplier;
            }
            if (position.unrealizedProfit === undefined || position.realizedProfit === undefined) {
                position.unrealizedProfit = (position.type === "POSITION_TYPE_BUY" ? 1 : -1) * (position.currentPrice - position.openPrice) * position.currentTickValue * position.volume / specification.tickSize;
                position.unrealizedProfit = Math.round(position.unrealizedProfit * multiplier) / multiplier;
                position.realizedProfit = position.profit - position.unrealizedProfit;
            }
            let newPositionPrice = position.type === "POSITION_TYPE_BUY" ? price.bid : price.ask;
            let isProfitable = (position.type === "POSITION_TYPE_BUY" ? 1 : -1) * (newPositionPrice - position.openPrice);
            let currentTickValue = isProfitable > 0 ? price.profitTickValue : price.lossTickValue;
            let unrealizedProfit = (position.type === "POSITION_TYPE_BUY" ? 1 : -1) * (newPositionPrice - position.openPrice) * currentTickValue * position.volume / specification.tickSize;
            unrealizedProfit = Math.round(unrealizedProfit * multiplier) / multiplier;
            position.unrealizedProfit = unrealizedProfit;
            position.profit = position.unrealizedProfit + position.realizedProfit;
            position.profit = Math.round(position.profit * multiplier) / multiplier;
            position.currentPrice = newPositionPrice;
            position.currentTickValue = currentTickValue;
        }
    }
    _filterRemovedPositions(positions) {
        return positions.filter((position)=>!this._combinedState.removedPositions[position.id]);
    }
    _filterRemovedOrders(orders) {
        return orders.filter((order)=>!this._combinedState.completedOrders[order.id]);
    }
    _getState(instanceIndex) {
        if (!this._stateByInstanceIndex["" + instanceIndex]) {
            this._logger.trace(`${this._account.id}:${instanceIndex}: constructed new state`);
            this._stateByInstanceIndex["" + instanceIndex] = this._constructTerminalState(instanceIndex);
        }
        return this._stateByInstanceIndex["" + instanceIndex];
    }
    _constructTerminalState(instanceIndex) {
        return {
            instanceIndex,
            connected: false,
            connectedToBroker: false,
            accountInformation: undefined,
            positions: [],
            orders: [],
            specificationsBySymbol: {},
            pricesBySymbol: {},
            ordersInitialized: false,
            positionsInitialized: false,
            lastSyncUpdateTime: 0,
            positionsHash: null,
            ordersHash: null,
            specificationsHash: null,
            isSpecificationsExpected: true,
            isPositionsExpected: true,
            isOrdersExpected: true,
            lastQuoteTime: undefined,
            lastQuoteBrokerTime: undefined
        };
    }
    /**
   * Constructs the instance of terminal state class
   * @param {MetatraderAccount} account mt account
   * @param {TerminalHashManager} terminalHashManager terminal hash manager
   * @param {MetaApiWebsocketClient} websocketClient websocket client
   */ constructor(account, terminalHashManager, websocketClient){
        super();
        this._id = _randomstring.default.generate(32);
        this._account = account;
        this._terminalHashManager = terminalHashManager;
        this._websocketClient = websocketClient;
        this._stateByInstanceIndex = {};
        this._waitForPriceResolves = {};
        this._combinedInstanceIndex = "combined";
        this._combinedState = {
            accountInformation: undefined,
            positions: [],
            orders: [],
            specificationsBySymbol: null,
            pricesBySymbol: {},
            removedPositions: {},
            completedOrders: {},
            specificationsHash: null,
            positionsHash: null,
            ordersHash: null,
            ordersInitialized: false,
            positionsInitialized: false,
            lastStatusTime: 0,
            lastQuoteTime: undefined,
            lastQuoteBrokerTime: undefined
        };
        this._processThrottledQuotesCalls = {};
        this._logger = _logger.default.getLogger("TerminalState");
        this._checkCombinedStateActivityJob = this._checkCombinedStateActivityJob.bind(this);
        this._checkCombinedStateActivityJobInterval = setInterval(this._checkCombinedStateActivityJob, 5 * 60 * 1000);
    }
};

//# sourceMappingURL=data:application/json;base64,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