'use strict';

import HttpClient from './clients/httpClient';
import DomainClient from './clients/domain.client';
import MetaStatsClient from './clients/metaStats.client';
import {RetryOptions} from './clients/httpClient';

/** connection options */
export type ConnectionOptions = {
  /** request timeout in seconds, default 60 */
  requestTimeout?: number,
  /** request domain, default 'agiliumtrade.agiliumtrade.ai' */
  domain?: string,
  /** retry options */
  retryOpts?: RetryOptions
}

/**
 * MetaStats API SDK
 */
export default class MetaStats {
  
  private _metaStatsClient: MetaStatsClient;

  /**
   * Constructs MetaStats class instance
   * @param {String} token authorization token
   * @param {ConnectionOptions} [opts] connection options
   */
  constructor(token: string, opts: ConnectionOptions = {}) {
    const httpClient = new HttpClient(opts.requestTimeout, opts.retryOpts);
    const domainClient = new DomainClient(httpClient, token, opts.domain);
    this._metaStatsClient = new MetaStatsClient(domainClient);
  }

  /**
   * Returns the getMetrics MetaStatsClient method bound to the MetaStatsClient instance
   * @returns {Function} getMetrics MetaStatsClient method
   */
  get getMetrics(): Function {
    return this._metaStatsClient.getMetrics.bind(this._metaStatsClient);
  }

  /**
   * Returns the getAccountTrades MetaStatsClient method bound to the MetaStatsClient instance
   * @returns {Function} getAccountTrades MetaStatsClient method
   */
  get getAccountTrades(): Function {
    return this._metaStatsClient.getAccountTrades.bind(this._metaStatsClient);
  }

  /**
   * Returns the getAccountOpenTrades MetaStatsClient method bound to the MetaStatsClient instance
   * @returns {Function} getAccountOpenTrades MetaStatsClient method
   */
  get getAccountOpenTrades(): Function {
    return this._metaStatsClient.getAccountOpenTrades.bind(this._metaStatsClient);
  }

  /**
   * Returns the resetMetrics MetaStatsClient method bound to the MetaStatsClient instance
   * @returns {Function} resetMetrics MetaStatsClient method
   */
  get resetMetrics(): Function {
    return this._metaStatsClient.resetMetrics.bind(this._metaStatsClient);
  }
}
